<?php
// Assumes $conn (database connection) and $_SESSION['user_id'] are available.

$user_id = $_SESSION['user_id'] ?? 1; // Default to ID 1 for safety
$delete_status = '';

// --- Handle Deletion ---
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['report_id'])) {
    $report_id_to_delete = (int)$_GET['report_id'];
    
    // Ensure the user owns the report before deleting (and it's not Approved/Pending if you want to restrict it)
    $sql_delete = "DELETE FROM monthly_activities WHERE report_id = ? AND submitted_by_user_id = ?";
    if ($stmt = $conn->prepare($sql_delete)) {
        $stmt->bind_param("ii", $report_id_to_delete, $user_id);
        if ($stmt->execute() && $stmt->affected_rows > 0) {
            $delete_status = "<div class='alert alert-success'>Report ID {$report_id_to_delete} deleted successfully.</div>";
        } else {
            $delete_status = "<div class='alert alert-error'>Error: Could not delete report or report not found.</div>";
        }
        $stmt->close();
    }
}

// --- Fetch Reports Submitted by the Current User ---
$reports = [];
$sql_reports = "
    SELECT 
        ma.report_id, 
        ma.report_month, 
        ma.title,
        ma.submission_date, 
        ma.status, 
        ma.status_date,
        u.username AS supervisor_name
    FROM monthly_activities ma
    JOIN users u ON ma.supervisor_user_id = u.user_id
    WHERE ma.submitted_by_user_id = ?
    ORDER BY ma.submission_date DESC";

if ($stmt = $conn->prepare($sql_reports)) {
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $reports[] = $row;
    }
    $stmt->close();
}

// Helper function to determine status color
function get_status_class($status) {
    switch ($status) {
        case 'Approved': return 'status-approved';
        case 'Pending': return 'status-pending';
        case 'Rejected': return 'status-rejected';
        case 'Draft': 
        default: return 'status-draft';
    }
}
?>

<style>
    .status-approved { color: #2ecc71; font-weight: bold; }
    .status-pending { color: #f39c12; font-weight: bold; }
    .status-rejected { color: #e74c3c; font-weight: bold; }
    .status-draft { color: #7f8c8d; }
    
    /* Simple table styling */
    .data-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 15px;
    }
    .data-table th, .data-table td {
        border: 1px solid #ecf0f1;
        padding: 12px;
        text-align: left;
    }
    .data-table th {
        background-color: #3498db;
        color: white;
    }
    .data-table tr:nth-child(even) {
        background-color: #f9f9f9;
    }
    .action-button {
        padding: 5px 10px;
        border: none;
        border-radius: 4px;
        cursor: pointer;
        font-size: 0.9em;
    }
    .btn-view { background-color: #3498db; color: white; }
    .btn-download { background-color: #f39c12; color: white; }
    .btn-delete { background-color: #e74c3c; color: white; }
</style>

<div class="form-card" style="margin-top: 20px;">
    <h3>Monthly Activity Report List</h3>
    <?php echo $delete_status; ?>
    
    <table class="data-table">
        <thead>
            <tr>
                <th>MONTH</th>
                <th>SUBMISSION DATE</th>
                <th>TITLE</th>
                <th>SUPERVISOR</th>
                <th>STATUS</th>
                <th>STATUS DATE</th>
                <th>ACTION</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($reports)): ?>
                <tr>
                    <td colspan="7" style="text-align: center;">No activity reports found.</td>
                </tr>
            <?php else: ?>
                <?php foreach ($reports as $report): ?>
                <tr>
                    <td><?php echo date('M Y', strtotime($report['report_month'])); ?></td>
                    <td><?php echo date('d-M-Y H:i A', strtotime($report['submission_date'])); ?></td>
                    <td><?php echo htmlspecialchars($report['title']); ?></td>
                    <td><?php echo htmlspecialchars($report['supervisor_name']); ?></td>
                    <td class="<?php echo get_status_class($report['status']); ?>">
                        <?php echo htmlspecialchars($report['status']); ?>
                    </td>
                    <td><?php echo $report['status_date'] ? date('d-M-Y', strtotime($report['status_date'])) : 'N/A'; ?></td>
                    <td>
                        <!-- View/Edit button (Drafts can be edited) -->
                        <button class="action-button btn-view" title="View/Edit">
                            <i class="fas fa-file-alt"></i>
                        </button>
                        <!-- Download button (mock for now) -->
                        <button class="action-button btn-download" title="Download PDF">
                            <i class="fas fa-download"></i>
                        </button>
                        <!-- Delete button -->
                        <a href="admin_dashboard.php?page=activity_report_list&action=delete&report_id=<?php echo $report['report_id']; ?>" 
                           onclick="return confirm('Are you sure you want to delete this report? This cannot be undone.');" 
                           class="action-button btn-delete" title="Delete">
                            <i class="fas fa-trash-alt"></i>
                        </a>
                    </td>
                </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>
