<?php
// FILE: create_activity_report.php
// Purpose: Allows an employee to create and submit a monthly activity report.

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
// ASSUMPTION: db_config.php is included here and provides the $conn variable.
// Make sure this path is correct relative to where you run this script.
include_once 'db_config.php'; 

$status_message = '';
// User ID must be retrieved from the session. Defaulting to 1 for demonstration.
$user_id = $_SESSION['user_id'] ?? 1; 

if (!isset($conn) || $conn->connect_error) {
    // If connection fails, display error and stop execution.
    $status_message = "<div class='alert alert-error'>Database connection error. Please check db_config.php.</div>";
    exit($status_message); 
}

// --- 1. Fetch Users for Supervisor Dropdown (Excludes the current user from being their own supervisor) ---
$users = [];
$sql_users = "SELECT user_id, username FROM users WHERE user_id != ? ORDER BY username ASC";
if ($stmt_users = $conn->prepare($sql_users)) {
    $stmt_users->bind_param("i", $user_id);
    $stmt_users->execute();
    $result_users = $stmt_users->get_result();
    while ($row = $result_users->fetch_assoc()) {
        $users[] = $row;
    }
    $stmt_users->close();
}

// --- 2. Handle Form Submission (Save Draft or Submit Pending) ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $report_month = $_POST['report_month'] ?? null;
    $supervisor_id = (int)($_POST['supervisor_id'] ?? 0);
    // Use real_escape_string or prepared statements for safety
    $title = $conn->real_escape_string($_POST['title'] ?? '');
    $description = $conn->real_escape_string($_POST['description'] ?? '');
    
    // Determine the status based on which button was clicked
    $status = isset($_POST['submit_report']) ? 'Pending' : 'Draft';
    $success_msg = ($status === 'Pending') 
        ? "Activity Report submitted successfully for review."
        : "Activity Report saved as draft.";

    // Input Validation
    if (!$report_month || $supervisor_id <= 0 || empty($title) || empty($description)) {
        $status_message = "<div class='alert alert-error'>Please fill in all required fields.</div>";
    } else {
        
        // Correct SQL: status_date and supervisor_feedback columns can be NULL on insertion
        $sql_insert = "INSERT INTO monthly_activities (submitted_by_user_id, supervisor_user_id, report_month, title, description, status, submission_date) 
                       VALUES (?, ?, ?, ?, ?, ?, NOW())";
        
        if ($stmt = $conn->prepare($sql_insert)) {
            // Binding types: (Integer, Integer, String, String, String, String) -> iissis
            $stmt->bind_param("iissis", $user_id, $supervisor_id, $report_month, $title, $description, $status);
            
            if ($stmt->execute()) {
                $status_message = "<div class='alert alert-success'>{$success_msg}</div>";
                // Optionally redirect the user after successful submission
            } else {
                $status_message = "<div class='alert alert-error'>Database error: " . $stmt->error . "</div>";
            }
            $stmt->close();
        } else {
            $status_message = "<div class='alert alert-error'>Database preparation error: " . $conn->error . "</div>";
        }
    }
}

$default_month = date('Y-m'); 
// Do NOT close connection here if other parts of admin_dashboard.php rely on it.
// If this file is standalone, it's safe to close: $conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Activity Report</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    <style>
        body { font-family: 'Inter', sans-serif; background-color: #f4f7f9; margin: 0; padding: 20px; }
        .form-card { max-width: 700px; margin: 20px auto; padding: 25px; border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.1); background-color: #fff; }
        label { display: block; margin-top: 15px; font-weight: bold; color: #555; }
        input[type="text"], input[type="month"], textarea, select { width: 100%; padding: 10px; margin-top: 5px; border: 1px solid #ddd; border-radius: 5px; box-sizing: border-box; }
        textarea { resize: vertical; }
        .alert { padding: 10px; margin-bottom: 15px; border-radius: 5px; font-weight: 600; }
        .alert-error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .alert-success { background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        button { cursor: pointer; border: none; padding: 10px 20px; border-radius: 5px; transition: opacity 0.2s; font-weight: 600; }
        button:hover { opacity: 0.9; }
    </style>
</head>
<body>
<div class="form-card">
    <h3>Create Monthly Activity Report (Employee ID: <?php echo htmlspecialchars($user_id); ?>)</h3>
    <?php echo $status_message; ?>

    <!-- IMPORTANT: Set the action to the correct path, likely your main dashboard with a parameter -->
    <form action="admin_dashboard.php?page=create_activity_report" method="post" id="activity_form">
        
        <div style="display: flex; gap: 20px; margin-bottom: 20px;">
            <div style="flex: 1;">
                <label for="report_month">SELECT MONTH *</label>
                <input type="month" name="report_month" id="report_month" value="<?php echo $default_month; ?>" required>
            </div>
            <div style="flex: 1;">
                <label for="supervisor_id">SUPERVISOR *</label>
                <select name="supervisor_id" id="supervisor_id" required>
                    <option value="">-- Select Supervisor --</option>
                    <?php 
                    foreach ($users as $user) {
                        echo '<option value="' . htmlspecialchars($user['user_id']) . '">' 
                           . htmlspecialchars($user['username']) 
                           . '</option>';
                    }
                    ?>
                </select>
            </div>
        </div>

        <label for="title">TITLE *</label>
        <input type="text" name="title" id="title" required placeholder="Project Status Update for the month">

        <label for="description">DESCRIPTION *</label>
        <textarea name="description" id="description" rows="6" required placeholder="Detailed list of activities completed this month..."></textarea>

        <div style="display: flex; gap: 10px; margin-top: 20px; justify-content: flex-start;">
            
            <button type="submit" name="submit_report" style="background-color: #2ecc71; color: white;">
                <i class="fas fa-check"></i> Submit (Pending)
            </button>
            
            <button type="submit" name="save_draft" style="background-color: #34495e; color: white;">
                <i class="fas fa-save"></i> Save Draft
            </button>
            
            <button type="button" onclick="window.location.href='admin_dashboard.php?page=activity_report_list'" style="background-color: #e74c3c; color: white;">
                <i class="fas fa-times"></i> Cancel
            </button>
        </div>
    </form>
</div>
</body>
</html>
